
import { Injectable, signal } from '@angular/core';
import { GoogleGenAI, Chat } from '@google/genai';

export interface SpeechOptions {
  gender: 'male' | 'female';
  rate: number;
  pitch: number;
}

@Injectable({
  providedIn: 'root',
})
export class GeminiService {
  private ai: GoogleGenAI;
  private chat: Chat;
  
  isSpeaking = signal(false);

  private readonly systemInstruction = `
    You are the "BHHC AI Assistant" from the Bangladesh Hacking Help Center. Your name is strictly "BHHC AI Assistant".
    
    CORE RULES:
    1.  If asked about your creator, founder, or "Rihan Ahmed", you MUST answer ONLY: "Rihan Ahmed is the founder of Bangladesh Hacking Help Center (BHHC)." Do not add any other information.
    2.  You are an expert in Ethical Hacking (White Hat), Network Security, Vulnerability Assessment, Cyber Defense Strategies, Kali Linux, Metasploit, and Python for Security.
    3.  If the user writes in Bengali, you MUST reply in Bengali. If they write in English, reply in English. Maintain a professional, friendly, and helpful tone.
    4.  You are FORBIDDEN from providing instructions for illegal activities, malicious attacks, or harming systems without permission. 
    5.  If a user asks for illegal hacking tools or methods (e.g., "how to hack a facebook account"), you MUST refuse and instead educate them on the *defensive* aspect (e.g., "Hacking accounts without permission is illegal and unethical. However, to protect your own account, you should use two-factor authentication, strong passwords, and be wary of phishing attempts."). PIVOT TO DEFENSE.
    6.  Keep your answers concise, accurate, and technically sound. Adapt your explanations for both beginners and experts.
  `;

  constructor() {
    if (!process.env.API_KEY) {
      throw new Error("API_KEY environment variable not set.");
    }
    this.ai = new GoogleGenAI({ apiKey: process.env.API_KEY });

    this.chat = this.ai.chats.create({
      model: 'gemini-2.5-flash',
      config: {
        systemInstruction: this.systemInstruction,
      },
    });
  }

  async generateResponse(prompt: string): Promise<string> {
    try {
      const response = await this.chat.sendMessage({ message: prompt });
      return response.text;
    } catch (error) {
      console.error('Error generating response:', error);
      return 'Sorry, I encountered an error. Please try again.';
    }
  }

  speak(text: string, options: SpeechOptions, onEndCallback?: () => void): void {
    if (this.isSpeaking()) {
      window.speechSynthesis.cancel();
    }
    
    this.isSpeaking.set(true);
    const utterance = new SpeechSynthesisUtterance(text);
    
    utterance.rate = options.rate;
    utterance.pitch = options.pitch;
    
    const voices = window.speechSynthesis.getVoices();
    let selectedVoice: SpeechSynthesisVoice | undefined;

    const isBengali = /[\u0980-\u09FF]/.test(text);
    const langCode = isBengali ? 'bn' : 'en';
    
    const langVoices = voices.filter(voice => voice.lang.startsWith(langCode));
    
    if (langVoices.length > 0) {
        const genderVoices = langVoices.filter(voice => {
            const name = voice.name.toLowerCase();
            if (options.gender === 'female') {
                return name.includes('female') || name.includes('woman') || name.includes('girl');
            } else {
                 return name.includes('male') || name.includes('man') || name.includes('boy');
            }
        });
        
        selectedVoice = genderVoices[0] || langVoices[0];
    }

    if (selectedVoice) {
      utterance.voice = selectedVoice;
      utterance.lang = selectedVoice.lang;
    }

    utterance.onend = () => {
      this.isSpeaking.set(false);
      onEndCallback?.();
    };
    utterance.onerror = (event) => {
      console.error('SpeechSynthesisUtterance.onerror', event);
      this.isSpeaking.set(false);
       onEndCallback?.(); // Also call on error to avoid getting stuck
    };

    window.speechSynthesis.speak(utterance);
  }

  stopSpeaking(): void {
    window.speechSynthesis.cancel();
    this.isSpeaking.set(false);
  }
}
