
import { Component, ChangeDetectionStrategy, signal, WritableSignal, inject, effect, viewChild, ElementRef, afterNextRender, computed } from '@angular/core';
import { FormsModule } from '@angular/forms';
import { GeminiService } from './services/gemini.service';
import { ChatMessage } from './models/chat.model';
import { LocalStorageService, VoiceSettings } from './services/local-storage.service';

// Declare the webkitSpeechRecognition for browsers that use the prefix.
declare var webkitSpeechRecognition: any;

type VoiceConversationState = 'listening' | 'thinking' | 'speaking' | 'idle';

@Component({
  selector: 'app-root',
  templateUrl: './app.component.html',
  changeDetection: ChangeDetectionStrategy.OnPush,
  standalone: true,
  imports: [FormsModule]
})
export class AppComponent {
  geminiService = inject(GeminiService);
  localStorageService = inject(LocalStorageService);
  
  messages: WritableSignal<ChatMessage[]> = signal<ChatMessage[]>([
    { id: 0, role: 'model', text: 'Hey Welcome I am Your BHHC Ai Assistant How Can I Help You' }
  ]);
  userInput = signal('');
  isLoading = signal(false);
  error = signal<string | null>(null);
  
  chatContainer = viewChild<ElementRef<HTMLDivElement>>('chatContainer');
  voiceTranscriptContainer = viewChild<ElementRef<HTMLDivElement>>('voiceTranscriptContainer');
  
  isSpeaking = this.geminiService.isSpeaking;
  currentlySpeakingMessageId = signal<number | null>(null);

  // Voice recognition signals
  isRecording = signal(false);
  isSpeechRecognitionSupported = signal(false);
  private recognition: any;
  
  // Voice assistant settings
  settingsOpen = signal(false);
  selectedGender = signal<'male' | 'female'>('female');
  speechRate = signal(1);
  speechPitch = signal(1);

  // Voice Conversation Mode
  isVoiceConversationActive = signal(false);
  voiceConversationState = signal<VoiceConversationState>('idle');
  voiceTranscript = signal('');
  voiceConversationHistory: WritableSignal<ChatMessage[]> = signal([]);

  voiceConversationStatusText = computed(() => {
    switch(this.voiceConversationState()) {
      case 'listening': return 'Listening...';
      case 'thinking': return 'Thinking...';
      case 'speaking': return 'Speaking...';
      default: return 'Starting voice chat...';
    }
  });

  constructor() {
    this.loadSettings();
    this.initializeSpeechRecognition();

    afterNextRender(() => {
        window.speechSynthesis.getVoices();
    });
    
    effect(() => {
      if (this.messages() && this.chatContainer()) {
        this.scrollToBottom();
      }
    });

    effect(() => {
        if (!this.isSpeaking()) {
            this.currentlySpeakingMessageId.set(null);
        }
    });

    effect(() => {
      const settings: VoiceSettings = {
        gender: this.selectedGender(),
        rate: this.speechRate(),
        pitch: this.speechPitch()
      };
      this.localStorageService.saveSettings(settings);
    });

    effect(() => {
      if (this.voiceConversationHistory() && this.voiceTranscriptContainer()) {
        this.scrollToVoiceBottom();
      }
    });
  }

  private loadSettings(): void {
    const savedSettings = this.localStorageService.loadSettings();
    if (savedSettings) {
      this.selectedGender.set(savedSettings.gender);
      this.speechRate.set(savedSettings.rate);
      this.speechPitch.set(savedSettings.pitch);
    }
  }

  private initializeSpeechRecognition(): void {
    try {
      const SpeechRecognition = (window as any).SpeechRecognition || webkitSpeechRecognition;
      if (SpeechRecognition) {
        this.isSpeechRecognitionSupported.set(true);
        this.recognition = new SpeechRecognition();
        this.recognition.continuous = false; // Set to false for conversation turn-taking
        this.recognition.interimResults = true;
        this.recognition.lang = navigator.language.startsWith('bn') ? 'bn-BD' : 'en-US';
        this.setupRecognitionListeners();
      }
    } catch (e) {
      console.error("Speech Recognition is not supported by this browser.", e);
      this.isSpeechRecognitionSupported.set(false);
    }
  }

  private setupRecognitionListeners(): void {
    this.recognition.onstart = () => {
      if (!this.isVoiceConversationActive()) {
        this.isRecording.set(true);
      }
    };
    this.recognition.onend = () => {
      if (this.isRecording()) {
        this.isRecording.set(false);
      }
      // In voice conversation mode, the logic is handled by onresult
    };
    this.recognition.onerror = (event: any) => {
      console.error('Speech recognition error:', event.error);
      this.isRecording.set(false);
      if (this.isVoiceConversationActive()) {
        this.endVoiceConversation(); // End on error to avoid getting stuck
      }
    };
    this.recognition.onresult = (event: any) => {
      let interimTranscript = '';
      let finalTranscript = '';

      for (let i = event.resultIndex; i < event.results.length; ++i) {
        if (event.results[i].isFinal) {
          finalTranscript += event.results[i][0].transcript;
        } else {
          interimTranscript += event.results[i][0].transcript;
        }
      }

      const currentTranscript = finalTranscript || interimTranscript;
      
      if (this.isVoiceConversationActive()) {
        this.voiceTranscript.set(currentTranscript);
        if (finalTranscript) {
          this.handleVoiceInput(finalTranscript.trim());
        }
      } else {
         this.userInput.set(currentTranscript);
      }
    };
  }
  
  toggleVoiceRecognition(): void {
    if (!this.isSpeechRecognitionSupported()) return;
    if (this.isRecording()) {
      this.recognition.stop();
    } else {
      this.userInput.set('');
      this.recognition.continuous = true; // for textarea mode
      this.recognition.start();
    }
  }

  async sendMessage(): Promise<void> {
    const messageText = this.userInput().trim();
    if (!messageText || this.isLoading()) return;

    if (this.isRecording()) {
        this.recognition.stop();
    }

    this.isLoading.set(true);
    this.error.set(null);
    const userMessage: ChatMessage = { id: Date.now(), role: 'user', text: messageText };
    this.messages.update(msgs => [...msgs, userMessage]);
    this.userInput.set('');
    
    const placeholderId = Date.now() + 1;
    const modelPlaceholder: ChatMessage = { id: placeholderId, role: 'model', text: '...' };
    this.messages.update(msgs => [...msgs, modelPlaceholder]);

    try {
      const responseText = await this.geminiService.generateResponse(messageText);
      this.messages.update(msgs => 
        msgs.map(msg => msg.id === placeholderId ? { ...msg, text: responseText } : msg)
      );
    } catch (e: any) {
      const errorMessage = 'Failed to get a response. Please check your connection or API key.';
      this.error.set(errorMessage);
       this.messages.update(msgs => 
        msgs.map(msg => msg.id === placeholderId ? { ...msg, text: errorMessage } : msg)
      );
    } finally {
      this.isLoading.set(false);
    }
  }

  toggleAudio(message: ChatMessage): void {
    if (this.currentlySpeakingMessageId() === message.id) {
        this.geminiService.stopSpeaking();
    } else {
        this.geminiService.speak(message.text, {
          gender: this.selectedGender(),
          rate: this.speechRate(),
          pitch: this.speechPitch()
        });
        this.currentlySpeakingMessageId.set(message.id);
    }
  }

  toggleSettings(): void {
    this.settingsOpen.update(value => !value);
  }

  trackByMessage(index: number, message: ChatMessage): number {
    return message.id;
  }

  private scrollToBottom(): void {
    try {
        const container = this.chatContainer()?.nativeElement;
        if (container) {
            container.scrollTop = container.scrollHeight;
        }
    } catch (err) {
      console.error('Could not scroll to bottom:', err);
    }
  }

  private scrollToVoiceBottom(): void {
     try {
        const container = this.voiceTranscriptContainer()?.nativeElement;
        if (container) {
            container.scrollTop = container.scrollHeight;
        }
    } catch (err) {
      console.error('Could not scroll voice transcript to bottom:', err);
    }
  }

  // --- Voice Conversation Methods ---

  startVoiceConversation(): void {
    this.isVoiceConversationActive.set(true);
    this.voiceConversationHistory.set([]);
    this.listenForUser();
  }

  endVoiceConversation(): void {
    this.isVoiceConversationActive.set(false);
    this.voiceConversationState.set('idle');
    this.recognition.stop();
    this.geminiService.stopSpeaking();
    if (this.voiceConversationHistory().length > 0) {
      this.messages.update(msgs => [...msgs, ...this.voiceConversationHistory()]);
    }
    this.voiceTranscript.set('');
  }

  private listenForUser(): void {
    if (!this.isVoiceConversationActive()) return;
    this.voiceConversationState.set('listening');
    this.voiceTranscript.set('');
    this.recognition.continuous = false; // Ensure turn-taking
    this.recognition.start();
  }
  
  private async handleVoiceInput(transcript: string): Promise<void> {
    if (!transcript || !this.isVoiceConversationActive()) {
      if(this.isVoiceConversationActive()) this.listenForUser();
      return;
    }

    this.voiceConversationState.set('thinking');
    const userMessage: ChatMessage = { id: Date.now(), role: 'user', text: transcript };
    this.voiceConversationHistory.update(history => [...history, userMessage]);

    try {
      const responseText = await this.geminiService.generateResponse(transcript);
      const modelMessage: ChatMessage = { id: Date.now() + 1, role: 'model', text: responseText };
      this.voiceConversationHistory.update(history => [...history, modelMessage]);
      this.voiceTranscript.set(responseText);
      
      this.voiceConversationState.set('speaking');
      this.geminiService.speak(
        responseText,
        {
          gender: this.selectedGender(),
          rate: this.speechRate(),
          pitch: this.speechPitch()
        },
        () => {
          this.listenForUser();
        }
      );
    } catch (e) {
      console.error("Error in voice conversation: ", e);
      const errorText = "Sorry, I ran into an error.";
      this.voiceConversationState.set('speaking');
      this.geminiService.speak(errorText, {
        gender: this.selectedGender(),
        rate: this.speechRate(),
        pitch: this.speechPitch()
      }, () => {
        this.endVoiceConversation();
      });
    }
  }
}
